import React, { useEffect, useState, type ReactNode } from "react";
import "./Permissions.css";
import SectionHeader from "../../layouts/SectionHeader";
import { useAppSelector, type RootState } from "../../stores/store";
import {
  IoAccessibilitySharp,
  IoApps,
  IoBatteryHalfOutline,
  IoCopyOutline,
  IoPhonePortraitOutline,
  IoWalkSharp,
} from "react-icons/io5";
import { FaMicrophone, FaUser } from "react-icons/fa";
import { FaLocationDot } from "react-icons/fa6";
import { IoMdNotifications } from "react-icons/io";
import { useTranslation } from "react-i18next";

interface IPermission {
  id: string;
  title: string;
  description: string;
  icon: ReactNode;
  permission: string;
  isGranted: boolean;
}

const Permissions: React.FC = () => {
  const [permissions, setPermissions] = useState<IPermission[]>([]);
  const { selectedDevice } = useAppSelector((state: RootState) => state.device);
  const { t } = useTranslation();

  const permissionMeta: Record<
    string,
    { title: string; description: string; icon: ReactNode }
  > = {
    INSTALLED_APPS: {
      title: t("permissions.installedApp"),
      description: t("permissions.installedAppDescription"),
      icon: <IoApps className="permission__icon" />,
    },
    RECORD_AUDIO: {
      title: t("permissions.microphone"),
      description: t("permissions.microphoneDescription"),
      icon: <FaMicrophone className="permission__icon" />,
    },
    ACTIVITY_RECOGNITION: {
      title: t("permissions.activity"),
      description: t("permissions.activityDescription"),
      icon: <IoWalkSharp className="permission__icon" />,
    },
    BACKGROUND_LOCATION: {
      title: t("permissions.location"),
      description: t("permissions.locationDescription"),
      icon: <FaLocationDot className="permission__icon" />,
    },
    ACCESSIBILITY_SERVICE: {
      title: t("permissions.accessibility"),
      description: t("permissions.accessibilityDescription"),
      icon: <IoAccessibilitySharp className="permission__icon" />,
    },
    SYSTEM_ALERT_WINDOW: {
      title: t("permissions.systemAlert"),
      description: t("permissions.systemAlertDescription"),
      icon: <IoCopyOutline className="permission__icon" />,
    },
    IGNORE_BATTERY_OPTIMIZATIONS: {
      title: t("permissions.batteryOptimization"),
      description: t("permissions.batteryOptimizationDescription"),
      icon: <IoBatteryHalfOutline className="permission__icon" />,
    },
    DEVICE_ADMIN: {
      title: t("permissions.deviceAdmin"),
      description: t("permissions.deviceAdminDescription"),
      icon: <FaUser className="permission__icon" />,
    },
    POST_NOTIFICATIONS: {
      title: t("permissions.notifications"),
      description: t("permissions.notificationsDescription"),
      icon: <IoMdNotifications className="permission__icon" />,
    },
    READ_PHONE_STATE: {
      title: t("permissions.phoneState"),
      description: t("permissions.phoneStateDescription"),
      icon: <IoPhonePortraitOutline className="permission__icon" />,
    },
  };

  useEffect(() => {
    if (!selectedDevice?.deviceInfo?.permissions) return;

    const updatedPermissions: IPermission[] =
      selectedDevice.deviceInfo.permissions.map((perm, index) => {
        const meta = permissionMeta[perm.permission] || {
          title: perm.permission,
          description: "No description available",
          icon: <i className="icon-default" />,
        };

        return {
          id: `${index}`,
          title: meta.title,
          description: meta.description,
          icon: meta.icon,
          permission: perm.permission,
          isGranted: perm.isGranted,
        };
      });

    setPermissions(updatedPermissions);
  }, [selectedDevice]);

  return (
    <div className="permissions wrapper">
      <SectionHeader to="/home" />

      <div className="permissions__content">
        <h3 className="permissions__content__title">
          {t("permissions.title")}
        </h3>

        <div className="permissions__content__items">
          {permissions.map((permission) => {
            return (
              <div key={permission.id} className="permissions__content__item">
                <div className="permissions__content__item__header">
                  <div>
                    {permission.icon}
                    <p>{permission.title}</p>
                  </div>

                  <div>
                    {permission.isGranted ? (
                      <p className="text-success">{t("common.granted")}</p>
                    ) : (
                      <p className="text-danger">{t("common.denied")}</p>
                    )}
                  </div>
                </div>

                <p className="permission__content__item__desc">
                  {permission.description}
                </p>
              </div>
            );
          })}
        </div>
      </div>
    </div>
  );
};

export default Permissions;
