import React, { useState, useRef, useEffect } from "react";
import "./Header.css";
import {
  IoNotificationsOutline,
  IoPhonePortraitOutline,
  IoReload,
  IoSettingsOutline,
} from "react-icons/io5";
import CSelect from "../../components/CSelect";
import { GrMenu } from "react-icons/gr";
import { NavLink } from "react-router-dom";
import { MdLogout } from "react-icons/md";
import { RiHome3Line } from "react-icons/ri";
import { BsCreditCard } from "react-icons/bs";
import { useTranslation } from "react-i18next";
import {
  useAppDispatch,
  useAppSelector,
  type RootState,
} from "../../stores/store";
import { accountSession } from "../../stores/slices/accountSlice";
import { FaUserAlt } from "react-icons/fa";

const devices = [
  {
    label: "iPhone 15 Pro",
    value: "iphone-15-pro",
    image: <IoPhonePortraitOutline />,
  },
  {
    label: "iPhone 14 Pro",
    value: "iphone-14-pro",
    image: <IoPhonePortraitOutline />,
  },
  {
    label: "iPhone 13 Pro",
    value: "iphone-13-pro",
    image: <IoPhonePortraitOutline />,
  },
];

const Header: React.FC = () => {
  const [selectedDevice, setSelectedDevice] = useState(devices[0].value);
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const sidebarRef = useRef<HTMLDivElement>(null);
  const { t } = useTranslation();
  const dispatch = useAppDispatch();
  const { session } = useAppSelector((state: RootState) => state.account);

  useEffect(() => {
    dispatch(accountSession());
  }, [dispatch]);

  const handleOutsideClick = (e: MouseEvent) => {
    if (sidebarRef.current && !sidebarRef.current.contains(e.target as Node)) {
      setIsSidebarOpen(false);
    }
  };

  useEffect(() => {
    if (isSidebarOpen) {
      document.addEventListener("mousedown", handleOutsideClick);
    } else {
      document.removeEventListener("mousedown", handleOutsideClick);
    }

    return () => {
      document.removeEventListener("mousedown", handleOutsideClick);
    };
  }, [isSidebarOpen]);

  return (
    <>
      <div className="header wrapper">
        <div className="header__actions">
          <GrMenu
            className="header__actions__icon"
            onClick={() => setIsSidebarOpen(true)}
          />

          <IoReload
            className="header__actions__icon"
            onClick={() => window.location.reload()}
          />
        </div>

        <div className="header__devices">
          <CSelect
            options={devices}
            value={selectedDevice}
            onChange={setSelectedDevice}
            style={{ width: "250px" }}
          />
        </div>
      </div>

      {isSidebarOpen && (
        <div className="sidebar__overlay">
          <div className="sidebar" ref={sidebarRef}>
            <div className="sidebar__content">
              <div className="sidebar__content__info">
                <FaUserAlt className="sidebar__content__info__icon" />

                <div className="text-bold">
                  <p>{session.login}</p>
                  <p>
                    {t("session.activeTill")}:{" "}
                    {session.subDateEnd ? (
                      <span className="text-success">{session.subDateEnd}</span>
                    ) : (
                      <span className="text-danger">
                        {t("session.noSubscription")}
                      </span>
                    )}
                  </p>
                  <p>
                    {t("session.emailStatus")}:{" "}
                    {session.emailVerified ? (
                      <span className="text-success">
                        {t("session.verified")}
                      </span>
                    ) : (
                      <span className="text-danger">
                        {t("session.unVerified")}
                      </span>
                    )}
                  </p>
                </div>
              </div>

              <div className="sidebar__content__navs">
                <NavLink
                  to={"/home"}
                  className={({ isActive }) => (isActive ? "active" : "")}
                  onClick={() => setIsSidebarOpen(false)}
                >
                  <RiHome3Line className="sidebard__content__navs__icon" />
                  <span>{t("navs.home")}</span>
                </NavLink>

                <NavLink
                  to={"/notifications"}
                  className={({ isActive }) => (isActive ? "active" : "")}
                  onClick={() => setIsSidebarOpen(false)}
                >
                  <IoNotificationsOutline className="sidebard__content__navs__icon" />
                  <span>{t("navs.notifications")}</span>
                </NavLink>

                <NavLink
                  to={"/subscription"}
                  className={({ isActive }) => (isActive ? "active" : "")}
                  onClick={() => setIsSidebarOpen(false)}
                >
                  <BsCreditCard className="sidebard__content__navs__icon" />
                  <span>{t("navs.subscriptions")}</span>
                </NavLink>

                <NavLink
                  to={"/settings"}
                  className={({ isActive }) => (isActive ? "active" : "")}
                  onClick={() => setIsSidebarOpen(false)}
                >
                  <IoSettingsOutline className="sidebard__content__navs__icon" />
                  <span>{t("navs.settings")}</span>
                </NavLink>
              </div>
            </div>

            <div className="sidebar__logout">
              <MdLogout className="sidebar__logout__icon" />
              <span>{t("navs.logout")}</span>
            </div>
          </div>
        </div>
      )}
    </>
  );
};

export default Header;
