import React, { useCallback, useEffect, useState } from "react";
import "./BlockApps.css";
import SectionHeader from "../../layouts/SectionHeader";
import { LuWalletCards } from "react-icons/lu";
import { GoLock, GoUnlock } from "react-icons/go";
import CButton from "../../components/CButton";
import { useTranslation } from "react-i18next";
import {
  useAppDispatch,
  useAppSelector,
  type RootState,
} from "../../stores/store";
import {
  fetchAppList,
  fetchBlackList,
  fetchWhiteList,
} from "../../stores/slices/appsSlice";
import CLoading from "../../components/CLoading";

const BlockApps: React.FC = () => {
  const [selectedTab, setSelectedTab] = useState<"all" | "blocked" | "allowed">(
    "all"
  );
  const { t } = useTranslation();
  const { selectedDevice } = useAppSelector((state: RootState) => state.device);
  const {
    allApps,
    isLoadingAllApps,
    blackList,
    isLoadingBlackList,
    whiteList,
    isLoadingWhiteList,
  } = useAppSelector((state: RootState) => state.apps);
  const dispatch = useAppDispatch();

  const fetchDatas = useCallback(async () => {
    const deviceId = selectedDevice?.id || "";

    await dispatch(fetchAppList(deviceId));
    await dispatch(fetchBlackList(deviceId));
    await dispatch(fetchWhiteList(deviceId));
  }, [selectedDevice?.id, dispatch]);

  useEffect(() => {
    fetchDatas();
  }, [fetchDatas]);

  const renderTabContent = () => {
    switch (selectedTab) {
      case "all":
        return (
          <div className="blockapps__content">
            <h3 className="blockapps__content__title">
              {t("blockapps.allapps")} ({allApps.length})
            </h3>
            <p className="blockapps__content__desc">
              {t("blockapps.allappsDesc")}
            </p>

            <div className="blockapps__content__actions">
              <CButton title={t("blockapps.block")} disabled />
              <CButton title={t("blockapps.allow")} disabled />
            </div>

            {!isLoadingAllApps && allApps.length === 0 && (
              <div className="blockapps__content__empty">
                <p>{t("blockapps.noFoundApps")}</p>
              </div>
            )}

            {isLoadingAllApps && (
              <div className="cloading__center">
                <CLoading />
              </div>
            )}

            <div className="blockapps__content__apps">
              {allApps.map((app, index) => {
                return (
                  <div key={index} className="blockapps__content__app">
                    <img
                      src={`data:image/jpeg;base64,${app.appIcon}`}
                      alt={app.appName}
                    />
                    <h4>{app.appName}</h4>
                  </div>
                );
              })}
            </div>
          </div>
        );
      case "blocked":
        return (
          <div className="blockapps__content">
            <h3 className="blockapps__content__title">
              {t("blockapps.blockedApps")} ({blackList.length})
            </h3>

            <div className="blockapps__content__actions">
              <CButton title={t("blockapps.unBlock")} disabled />
            </div>

            {!isLoadingBlackList && blackList.length === 0 && (
              <div className="blockapps__content__empty">
                <p>{t("blockapps.noFoundApps")}</p>
              </div>
            )}

            {isLoadingBlackList && (
              <div className="cloading__center">
                <CLoading />
              </div>
            )}

            <div className="blockapps__content__apps">
              {blackList.map((app, index) => {
                return (
                  <div key={index} className="blockapps__content__app">
                    <img
                      src={`data:image/jpeg;base64,${app.appIcon}`}
                      alt={app.appName}
                    />
                    <h4>{app.appName}</h4>
                  </div>
                );
              })}
            </div>
          </div>
        );
      case "allowed":
        return (
          <div className="blockapps__content">
            <h3 className="blockapps__content__title">
              {t("blockapps.allowedApps")} ({whiteList.length})
            </h3>

            <div className="blockapps__content__actions">
              <CButton title={t("blockapps.remove")} disabled />
            </div>

            {!isLoadingWhiteList && whiteList.length === 0 && (
              <div className="blockapps__content__empty">
                <p>{t("blockapps.noFoundApps")}</p>
              </div>
            )}

            {isLoadingWhiteList && (
              <div className="cloading__center">
                <CLoading />
              </div>
            )}

            <div className="blockapps__content__apps">
              {whiteList.map((app, index) => {
                return (
                  <div key={index} className="blockapps__content__app">
                    <img
                      src={`data:image/jpeg;base64,${app.appIcon}`}
                      alt={app.appName}
                    />
                    <h4>{app.appName}</h4>
                  </div>
                );
              })}
            </div>
          </div>
        );
      default:
        return null;
    }
  };

  return (
    <div className="blockapps wrapper">
      <SectionHeader to="/home/manage-apps" />

      <div>{renderTabContent()}</div>

      <div className="blockapps__tabs">
        <div
          onClick={() => setSelectedTab("all")}
          className={`blockapps__tab ${selectedTab === "all" ? "active" : ""}`}
        >
          <LuWalletCards />
          <p>{t("blockapps.allapps")}</p>
        </div>

        <div
          onClick={() => setSelectedTab("blocked")}
          className={`blockapps__tab ${
            selectedTab === "blocked" ? "active" : ""
          }`}
        >
          <GoLock />
          <p>{t("blockapps.blockedApps")}</p>
        </div>

        <div
          onClick={() => setSelectedTab("allowed")}
          className={`blockapps__tab ${
            selectedTab === "allowed" ? "active" : ""
          }`}
        >
          <GoUnlock />
          <p>{t("blockapps.allowedApps")}</p>
        </div>
      </div>
    </div>
  );
};

export default BlockApps;
