import React, { useEffect, useState } from "react";
import "./BlockApps.css";
import SectionHeader from "../../layouts/SectionHeader";
import { LuWalletCards } from "react-icons/lu";
import { GoLock, GoUnlock } from "react-icons/go";
import CButton from "../../components/CButton";
import { useTranslation } from "react-i18next";
import {
  useAppDispatch,
  useAppSelector,
  type RootState,
} from "../../stores/store";
import { fetchAppList } from "../../stores/slices/appsSlice";

const BlockApps: React.FC = () => {
  const [selectedTab, setSelectedTab] = useState<"all" | "blocked" | "allowed">(
    "all"
  );
  const { t } = useTranslation();
  const { selectedDevice } = useAppSelector((state: RootState) => state.device);
  const { allApps } = useAppSelector((state: RootState) => state.apps);
  const dispatch = useAppDispatch();

  useEffect(() => {
    dispatch(fetchAppList(selectedDevice?.id || ""));
  }, [selectedDevice, dispatch]);

  const renderTabContent = () => {
    switch (selectedTab) {
      case "all":
        return (
          <div className="blockapps__content">
            <h3 className="blockapps__content__title">
              {t("blockapps.allapps")} ({allApps.length})
            </h3>
            <p className="blockapps__content__desc">
              {t("blockapps.allappsDesc")}
            </p>

            <div className="blockapps__content__actions">
              <CButton title={t("blockapps.block")} disabled />
              <CButton title={t("blockapps.allow")} disabled />
            </div>

            {allApps.length === 0 && (
              <div className="blockapps__content__empty">
                <p>{t("blockapps.noFoundApps")}</p>
              </div>
            )}

            <div className="blockapps__content__apps">
              {allApps.map((app, index) => {
                return (
                  <div key={index} className="blockapps__content__app">
                    <img
                      src={`data:image/jpeg;base64,${app.appIcon}`}
                      alt={app.appName}
                    />
                    <h4>{app.appName}</h4>
                  </div>
                );
              })}
            </div>
          </div>
        );
      case "blocked":
        return (
          <div className="blockapps__content">
            <h3 className="blockapps__content__title">
              {t("blockapps.blockedApps")} (0)
            </h3>

            <div className="blockapps__content__actions">
              <CButton title={t("blockapps.unBlock")} disabled />
            </div>

            <div className="blockapps__content__empty">
              <p>{t("blockapps.noFoundApps")}</p>
            </div>

            <div className="blockapps__content__apps"></div>
          </div>
        );
      case "allowed":
        return (
          <div className="blockapps__content">
            <h3 className="blockapps__content__title">
              {t("blockapps.allowedApps")} (0)
            </h3>

            <div className="blockapps__content__actions">
              <CButton title={t("blockapps.remove")} disabled />
            </div>

            <div className="blockapps__content__empty">
              <p>{t("blockapps.noFoundApps")}</p>
            </div>

            <div className="blockapps__content__apps"></div>
          </div>
        );
      default:
        return null;
    }
  };

  return (
    <div className="blockapps wrapper">
      <SectionHeader to="/home" />

      <div>{renderTabContent()}</div>

      <div className="blockapps__tabs">
        <div
          onClick={() => setSelectedTab("all")}
          className={`blockapps__tab ${selectedTab === "all" ? "active" : ""}`}
        >
          <LuWalletCards />
          <p>{t("blockapps.allapps")}</p>
        </div>

        <div
          onClick={() => setSelectedTab("blocked")}
          className={`blockapps__tab ${
            selectedTab === "blocked" ? "active" : ""
          }`}
        >
          <GoLock />
          <p>{t("blockapps.blockedApps")}</p>
        </div>

        <div
          onClick={() => setSelectedTab("allowed")}
          className={`blockapps__tab ${
            selectedTab === "allowed" ? "active" : ""
          }`}
        >
          <GoUnlock />
          <p>{t("blockapps.allowedApps")}</p>
        </div>
      </div>
    </div>
  );
};

export default BlockApps;
