import {
  createAsyncThunk,
  createSlice,
  type PayloadAction,
} from "@reduxjs/toolkit";
import { sendRpcRequest } from "../../services/apiClient";
import { TG_USER_ID } from "../../constants/constants";

interface IAuthUser {
  token: string;
}

interface IAuthState {
  user: IAuthUser | null;
  loading: boolean;
  error: string | null;
  success: boolean | string;
}

interface IResponseError {
  code: number;
  message: string;
}

const initialState: IAuthState = {
  user: null,
  loading: false,
  error: null,
  success: false,
};

export const authLogin = createAsyncThunk<
  IAuthUser,
  { login: string; password: string },
  { rejectValue: string }
>("auth/login", async ({ login, password }, { rejectWithValue }) => {
  try {
    const response = await sendRpcRequest("auth.loginPass", {
      login,
      password,
      fcmKey: null,
    });

    return response as IAuthUser;
  } catch (error: unknown) {
    const err = error as IResponseError;
    return rejectWithValue(err.message);
  }
});

export const authRegister = createAsyncThunk<
  IAuthUser,
  { login: string; password: string },
  { rejectValue: string }
>("auth/register", async ({ login, password }, { rejectWithValue }) => {
  try {
    const response = await sendRpcRequest("auth.registerloginpass", {
      login,
      password,
    });

    return response as IAuthUser;
  } catch (error: unknown) {
    const err = error as IResponseError;
    return rejectWithValue(err.message);
  }
});

export const authRecover = createAsyncThunk<
  IAuthUser,
  { login: string; newPassword: string },
  { rejectValue: string }
>("auth/recover", async ({ login, newPassword }, { rejectWithValue }) => {
  try {
    const response = await sendRpcRequest("auth.resetpassword", {
      login,
      newPassword,
    });

    return response as IAuthUser;
  } catch (error: unknown) {
    const err = error as IResponseError;
    return rejectWithValue(err.message);
  }
});

const authSlice = createSlice({
  name: "auth",
  initialState,
  reducers: {
    logout() {
      localStorage.removeItem(`token-${TG_USER_ID}`);
      localStorage.removeItem(`pincode-${TG_USER_ID}`);
      sessionStorage.removeItem(`pincode-${TG_USER_ID}`);
      window.location.href = "/";
    },
  },
  extraReducers: (builder) => {
    builder
      .addCase(authLogin.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(
        authLogin.fulfilled,
        (state, action: PayloadAction<IAuthUser>) => {
          state.loading = false;
          state.user = action.payload;
          state.success = true;
        }
      )
      .addCase(authLogin.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload ?? "Login failed";
      })

      .addCase(authRegister.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(
        authRegister.fulfilled,
        (state, action: PayloadAction<IAuthUser>) => {
          state.loading = false;
          state.user = action.payload;
          state.success = true;
        }
      )
      .addCase(authRegister.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload ?? "Registration failed";
      })

      .addCase(authRecover.pending, (state) => {
        state.loading = true;
        state.error = null;
      })
      .addCase(
        authRecover.fulfilled,
        (state, action: PayloadAction<IAuthUser>) => {
          state.loading = false;
          state.user = action.payload;
          state.success = "Confirmation code sent to email";
        }
      )
      .addCase(authRecover.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload ?? "Recovery failed";
      });
  },
});

export const { logout } = authSlice.actions;
export default authSlice.reducer;
