import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import { sendRpcRequest } from "../../services/apiClient";
import type { IDevice } from "../../types/device.types";

interface IDeviceState {
  devices: IDevice[];
  selectedDevice: IDevice | null;
  loadingDevices: boolean;
  errorDevices: string;
}

const initialState: IDeviceState = {
  devices: [],
  selectedDevice: null,
  loadingDevices: false,
  errorDevices: "",
};

export const fetchDevices = createAsyncThunk(
  "device/fetchDevices",
  async (_, { rejectWithValue }) => {
    try {
      const response = await sendRpcRequest<{ list: IDevice[] }>(
        "devices.getlist"
      );

      return response.list;
    } catch (error: unknown) {
      if (typeof error === "object" && error !== null && "message" in error) {
        return rejectWithValue(error.message);
      }

      return rejectWithValue("Unknown error occurred");
    }
  }
);

const deviceSlice = createSlice({
  name: "device",
  initialState,
  reducers: {
    setSelectedDevice: (state, action) => {
      const selected = state.devices.find((d) => d.id === action.payload);
      state.selectedDevice = selected ?? null;
    },
  },
  extraReducers: (builder) => {
    builder
      .addCase(fetchDevices.pending, (state) => {
        state.loadingDevices = true;
      })
      .addCase(fetchDevices.fulfilled, (state, action) => {
        state.loadingDevices = false;
        state.devices = action.payload;
        state.selectedDevice = action.payload[0] ?? null;
      })
      .addCase(fetchDevices.rejected, (state, action) => {
        state.loadingDevices = false;
        state.errorDevices = action.payload as string;
      });
  },
});

export const { setSelectedDevice } = deviceSlice.actions;
export default deviceSlice.reducer;
